<?php

/**
 * Class Menu - Menu Builder class pulling menu from database based
 * on the class and category of user
 *
 * @author OPMat
 */

//session_start();
class Menu extends Logger {
    
    /**
    * The database connection object.
    *
    * @var object
    */
    private $_db;
    
    /**
    * The User ID.
    *
    * @var integer
    */
    private $_uid;
    
    /**
    * The User Name.
    *
    * @var string
    */
    private $_username;
    
    /**
    * The User's Group(s).
    *
    * @var object  Array of all groups the user belong to
    */
    public $ugroup = [];
    
    /**
    * The User's Menu(s).
    *
    * @var object   Multi-dimensional Array of all Menu(s) the user has access to
    */
    public $umenus = [];
    
    public $isClient = 1;
    
    public $ip_addy = '0.0.0.0';
    
    private static $_blankArr = [0];
    
    private $_server = SERVER_URL;
    
    //replace %s with fullname, raduser, email
    private $_fixed_menu_client = <<<EQD
            <li>
          <a href="javascript:void(0)" onclick="lcjak_linkOpen(0,1,'en','%s (%s-web)','%s','');return false;" >
            <i class="fa fa-th"></i> <span>Live Chat</span>
            <span class="pull-right-container">
              <small class="label pull-right bg-green">new</small>
            </span>
          </a>
        </li>
        <li >
          <a href="%s">
            <i class="fa fa-dedent"></i>
            <span>Logout</span>
            <span class="pull-right-container">
              <i class="fa fa-angle-left pull-right"></i>
            </span>
          </a>
        </li>
EQD;
    
    private $_fixed_menu_admin = <<<EQD
        <li >
          <a href="%s">
            <i class="fa fa-dedent"></i>
            <span>Logout</span>
            <span class="pull-right-container">
              <i class="fa fa-angle-left pull-right"></i>
            </span>
          </a>
        </li>
EQD;
    
    /**
    * Class constructor.
    *
     * @param object $dbconnection Database Connection Object
     * @param integer $userid User ID (optional)
     * @param object $user_group One-Dimensional array of group(s) the user belongs to
    */
    public function __construct($dbconnection, $ipaddress, $username, $isClient = 1, $userid = 0, $user_group = [0]) {
        parent::__construct($dbconnection, $username);
        $this->_db = $dbconnection;
        $this->_uid = ($userid > 0)? $userid : $this->_uid;
        $this->ugroup = (count($user_group))? $user_group : $this->ugroup;
        $this->isClient = $isClient;
        $this->ip_addy = $ipaddress;
        $this->_username = $username;
    } 
    
    /**
     * Getter Method for Private Variable $_uid
     * 
     * @return integer returns the user id
     */
    public function getUID() {
        return $this->_uid;
    }
    
    /**
     * Setter Method for Private Variable $_uid
     * 
     * @param integer $uid User ID
     */
    public function setUID($uid) {
        $this->_uid = $uid;
    }
    
    /**
     * Getter Method for Private Variable $_username
     * 
     * @return string returns the username
     */
    public function getUsername() {
        return $this->_username;
    }
    
    /**
     * Setter Method for Private Variable $_username
     * 
     * @param integer $uid username
     */
    public function setUsername($username) {
        $this->_username = $username;
    }
    
    /**
     * Setter Method for Private Database Object $_db
     * 
     * @param object $db Database Object
     */
    public function setDB($db) {
        $this->_db = $db;
    }
    
    /**
     * Check if the User Group passed is set, if not, it checks if the user group
     * attribute is set. If not, it checks if the user id attribute is set and gets 
     * the user group based on the user id attribute
     * 
     * @param Object $user_group One-Dimensional array of group(s) the user belongs to
     * 
     * @return Object One-Dimensional array of group(s) the user belongs to
     */
    public function getUserGroup($user_group) {
        if ( (count($user_group)>0) && ($user_group != [0]) ) {
            return $user_group;
        } elseif ( (count($this->ugroup)>0) && ($this->ugroup != [0]) ) {
            return $this->ugroup;
        } elseif ( (isset($this->_uid)) && ($this->_uid > 0) ) {
            return $this->_getUserGroupFromID($this->_uid);
        } else {
            return [0];
        }
    }
    
    /**
     * Get User Group from DB based on the supplied user id
     * 
     * @param Object $userid One-Dimensional array of group(s) the user belongs to
     * 
     * @return Object One-Dimensional array of group(s) the user belongs to
     */
    private function _getUserGroupFromID($userid) {
        $ugrp = [];
        
        $strsql = "SELECT UGID FROM usergrp_rel WHERE USERID=?";
        $some_sth = $this->_db->select($strsql, 'i', $userid);
        $rows = $some_sth->num_rows();
        while ( $row = $some_sth->fetch_object() ) :
            array_push($ugrp, $row->UGID);
            //or $ugrp[] = $row->UGID;
        endwhile;
        
        $this->ugroup = $ugrp;
        
        return $ugrp;
    }
    
    /**
     * This method adds a new menu to the database
     * 
     * @param string $menu Menu Item to be added
     * @param integer $admin Admin User ID for the user creating the menu
     * @param integer $pmenu The Parent menu ID. Defaults to 0
     * @param integer $morder The Order number for the menu among its siblings, default = 1
     * @param integer $isClient Determines if a menu is meant strictly for students' interface or Admin/Staff. Default = 1, which means the menu is for Students' interface
     * @param string $murl The menu URL. Default to empty string
     * @param string $micon Menu Icon
     * @param string $mimg Menu Image
     * @param string $tooltip Menu Tooltip text
     * @param integer $active Determines if the menu should be set active or not. Default = 1, meaning active and 0 means inactive.
     * 
     * @return integer/boolean Returns value>=1/TRUE if menu is successfully added else returns 0/FALSE
     */
    public function addMenu($menu, $admin, $pmenu=0, $morder=1, $isClient=1, $murl="", $micon="", $mimg="", $tooltip="", $active=1) {
        if ($menu == "") :
            return "Error: Invalid Menu Name!";
        else:
            $ins = "INSERT INTO tbl_menus (MENU, CREATEDBY, PARENTMID, MENU_ORDER, IS_USER_MENU, URL, MENUICON, MENUIMAGE,
                    TOOLTIP, IS_ACTIVE, DATECREATED) VALUES (?,?,?,?,?,?,?,?,?,?, NOW()) ";
            $saved = $this->_db->insert($ins, "siiiissssi", $menu, $admin, $pmenu, $morder, $isClient, $murl, $micon, $mimg, $tooltip, $active);
        endif;
        
        if ($this->getUsername() == "") $this->setUsername ($admin);
        if ($saved > 0) {
            $this->logTrail(Logger::INFO, "$admin added MENU $menu with PARENT $pmenu and ORDER $morder ($murl, $micon, $mimg, $tooltip, $active)", Logger::STATUS_PASSED, $this->ip_addy);
            return 1;
        } else {
            $this->logTrail(Logger::INFO, "$admin tried adding MENU $menu with PARENT $pmenu and ORDER $morder ($murl, $micon, $mimg, $tooltip, $active)", Logger::STATUS_FAILED, $this->ip_addy);
            return 0;
        }
    }
    
    /**
     * This method updates a menu's parameter
     * 
     * @param integer $mid Menu ID for the menu to be updated
     * @param string $menu Menu Item to be added
     * @param integer $admin Admin User ID for the user creating the menu
     * @param integer $pmenu The Parent menu ID. Defaults to 0
     * @param integer $morder The Order number for the menu among its siblings, default = 1
     * @param integer $isClient Determines if a menu is meant strictly for students' interface or Admin/Staff. Default = 1, which means the menu is for Students' interface
     * @param string $murl The menu URL. Default to empty string
     * @param string $micon Menu Icon
     * @param string $mimg Menu Image
     * @param string $tooltip Menu Tooltip text
     * @param integer $active Determines if the menu should be set active or not. Default = 1, meaning active and 0 means inactive.
     * 
     * @return integer/boolean Returns value>=1/TRUE if menu is successfully updated else returns 0/FALSE
     */
    public function updateMenu($mid, $menu, $admin, $pmenu=0, $morder=1, $isClient=1, $murl="", $micon="", $mimg="", $tooltip="", $active=1) {
        if ($menu == "") :
            return "Error: Invalid Menu Name!";
        else:
            $upd = "UPDATE tbl_menus SET MENU=?, MODIFIEDBY=?, PARENTMID=?, MENU_ORDER=?, IS_USER_MENU=?, URL=?, MENUICON=?, 
                    MENUIMAGE=?, TOOLTIP=?, IS_ACTIVE=?, LASTMODIFIED=NOW() WHERE MID=?";
            $saved = $this->_db->execute_query($upd, "siiiissssi", $menu, $admin, $pmenu, $morder, $isClient, $murl, $micon, $mimg, $tooltip, $active, $mid);
        endif;
        
        if ($this->getUsername() == "") $this->setUsername ($admin);
        if ($saved){
            $this->logTrail(Logger::INFO, "$admin updated MENU $menu with PARENT $pmenu and ORDER $morder ($murl, $micon, $mimg, $tooltip, $active)", Logger::STATUS_PASSED, $this->ip_addy);
            return 1;
        } else {
            $this->logTrail(Logger::INFO, "$admin tried updating MENU $menu with PARENT $pmenu and ORDER $morder ($murl, $micon, $mimg, $tooltip, $active)", Logger::STATUS_FAILED, $this->ip_addy);
            return 0;
        }
    }
    
    /**
     * Update Menu Order with the new Order
     * 
     * @param string $menustr Sorted Menu JSON string
     * @param string $admin Admin Username
     * 
     * @return integer/boolean Returns the status of the update transaction 1/TRUE if successful else 0/FALSE
     */
    public function updateMenuOrder($menus, $admin="") {
//        [{"id":1},{"id":4,"children":[{"id":6}]},{"id":9},{"id":2,"children":[{"id":3},{"id":8},{"id":7},{"id":10}]},{"id":12},{"id":11}]
        $marr = json_decode($menus, TRUE);
        
        $ct = 0;
        $this->_db->autocommit(FALSE);
        $all_query_ok=true; // our control variable
        foreach ($marr as $key => $value) {
            $upd = "UPDATE tbl_menus SET MENU_ORDER=? WHERE MID=?";
            $saved = $this->_db->execute_query($upd, "ii", ++$ct, $value["id"]);
            if (!$saved) $all_query_ok = FALSE;
            if (isset($value["children"]) && (count($value["children"])) ):
                $sbct = 0;
                foreach ($value["children"] as $val) {
                    $upd = "UPDATE tbl_menus SET MENU_ORDER=? WHERE MID=?";
                    $saved = $this->_db->execute_query($upd, "ii", ++$sbct, $val["id"]);
                    if (!$saved) $all_query_ok = FALSE;
                }
            endif;
        }
        
        $all_query_ok ? $this->_db->commit() : $this->_db->rollback();
        $this->_db->autocommit(TRUE);
        if ($all_query_ok) :   
            $this->logTrail(Logger::INFO, "$admin updated MENU ORDER", Logger::STATUS_PASSED, $this->ip_addy);
            return 1;
        else: 
            $this->logTrail(Logger::INFO, "$admin tried updating MENU ORDER", Logger::STATUS_FAILED, $this->ip_addy);
            return 0;
        endif;
        
    }
    
    /**
     * Recursive Menu Order module
     * 
     * @param mixed $marr Sorted Menu Array
     * @param boolean $all_query_ok Query status passed by reference
     * 
     */
    private function _saveSort($marr, &$all_query_ok) {
        //$all_query_ok=true; // our control variable
        $ct = 0;
        foreach ($marr as $key => $value) {
            $upd = "UPDATE tbl_menus SET MENU_ORDER=? WHERE MID=?";
            $saved = $this->_db->execute_query($upd, "ii", ++$ct, $value["id"]);
            if (!$saved) $all_query_ok = FALSE;
            if (isset($value["children"]) && (count($value["children"])) ):
                $this->_saveSort($value["children"], $all_query_ok);
            endif;
        }
        
        return $all_query_ok;
    }
    
    /**
     * Update Menu Order with the new Order (Rewrite with Recursion)
     * 
     * @param string $menustr Sorted Menu JSON string
     * @param string $admin Admin Username
     * 
     * @return integer/boolean Returns the status of the update transaction 1/TRUE if successful else 0/FALSE
     */
    public function updateMenuOrder2($menus, $admin="") {
//        [{"id":1},{"id":4,"children":[{"id":6}]},{"id":9},{"id":2,"children":[{"id":3},{"id":8},{"id":7},{"id":10}]},{"id":12},{"id":11}]
        $marr = json_decode($menus, TRUE);
        
        $ct = 0;
        $this->_db->autocommit(FALSE);
        $all_query_ok = TRUE;
        
        $this->_saveSort($marr, $all_query_ok);
        
        $all_query_ok ? $this->_db->commit() : $this->_db->rollback();
        $this->_db->autocommit(TRUE);
        if ($all_query_ok) :   
            $this->logTrail(Logger::INFO, "$admin updated MENU ORDER", Logger::STATUS_PASSED, $this->ip_addy);
            return 1;
        else: 
            $this->logTrail(Logger::INFO, "$admin tried updating MENU ORDER", Logger::STATUS_FAILED, $this->ip_addy);
            return 0;
        endif;
        
    }
    
    
    /**
     * This method loads menu for a category of users (Staff/Admin or Students) for ordering in the Nestable plugin format
     * 
     * @param integer $isActiveOnly Determines if only active menus are to be loaded or to load all menus
     * @param integer $isClient Determines if to load Students' menu or that of Admin/Staff. Loads students' menu if set to 1
     *      * 
     * @return string Returns the menu items in the <a href="https://github.com/dbushell/Nestable#usage">Nestable</a> plugin format
     * 
     * @link https://github.com/dbushell/Nestable#usage Nestable
     */
    public function loadNestableMenus($isActiveOnly=0, $isClient=1) {
        //Load Menus
        $menus = $this->loadMenusAdmin($isActiveOnly, $isClient);
        
        //Menu to Nestable format
        return $this->_menu2Nestable($menus);
    }
    
    /**
     * Convert Menu Object to Nestable format
     * 
     * @param mixed $menus Menu Object
     * 
     * @return string Netable Menu string
     */
    private function _menu2Nestable($menus) {
        
        $menustr = '
                <div class="dd">
                    <ol class="dd-list">';
        foreach ($menus as $key => $value) {
            $menustr .= "<li class=\"dd-item\" data-id=\"{$value["MID"]}\">";
            $menustr .= "<div class=\"dd-handle\">{$value["MENU"]}</div>";
            
            if (isset($value["SUBMENU"]) && count($value["SUBMENU"])):
                $menustr .= $this->_submenu2Nestable($value["SUBMENU"]);
            endif;
            
            $menustr .= "</li>"; 
        }
        
        $menustr .= '
                    </ol>
                </div>';
        
        return $menustr;
    }
    
    /**
     * Convert SubMenu Object to Nestable format
     * 
     * @param mixed $menus Menu Object
     * 
     * @return string Netable Menu string
     */
    private function _submenu2Nestable($menus) {
        $menustr = '
                    <ol class="dd-list">';
        foreach ($menus as $key => $value) {
            $menustr .= "<li class=\"dd-item\" data-id=\"{$value["MID"]}\">";
            $menustr .= "<div class=\"dd-handle\">{$value["MENU"]}</div>";
            
            if (isset($value["SUBMENU"]) && count($value["SUBMENU"])):
                $menustr .= $this->_submenu2Nestable($value["SUBMENU"]);
            endif;
            
            $menustr .= "</li>"; 
        }
        
        $menustr .= '
                    </ol>
                    ';
        
        return $menustr;
    }
    
    
    /**
     * This method loads side menu for a users (Staff/Admin or Students) based on $user_group in the Side Menu format
     * 
     * @param object $user_group One-Dimensional array of group(s) the user belongs to
     * @param string $menuheader The Menu Header string. Default "MAIN NAVIGATION"
     * 
     * @return string Menu String for display as side menu for a user or user group
     */
    public function loadSideMenus($user_group, $menuheader="MAIN NAVIGATION", $cur_menu="") {
        //Load Menus
        $menus = $this->loadMenus($user_group); //print_r($menus);
        
        //get ParentID
        $pid = $this->getParentMenuID($cur_menu);
        
        //Menu to Nestable format
        return $this->_menu2SideMenu($menus, $menuheader, $cur_menu, $pid);
    }
    
    /**
     * Convert Menu Object to Side Menu AdminLTE format
     * 
     * @param mixed $menus Menu Object
     * @param string $menuheader The Menu Header string. Default "MAIN NAVIGATION"
     * 
     * @return string Side Menu AdminLTE format string
     */
    private function _menu2SideMenu($menus, $menuheader="MAIN NAVIGATION", $cur_menu="", $pid=0) {
        
        $menustr = '
                <ul class="sidebar-menu" data-widget="tree">
                    <li class="header">' . $menuheader . '</li>';

        foreach ($menus as $key => $value) {
            
            $ico = ($value["MENUICON"] != "")? "<i class=\"{$value["MENUICON"]}\"></i>" : "";
            $subico = (isset($value["SUB_MENU"]) && count($value["SUB_MENU"]))? "<span class=\"pull-right-container\"><i class=\"fa fa-angle-left pull-right\"></i></span>" : "";
            
            if (isset($value["SUB_MENU"]) && count($value["SUB_MENU"])):
                $sty = ($pid == $value["MID"])? "active menu-open" : "";
                $menustr .= "<li class=\"treeview $sty\">
                    <a href=\"{$this->_server}{$value["URL"]}\"> $ico
                        <span>{$value["MENU"]}</span>
                        $subico
                    </a>";
                
                $menustr .= $this->_submenu2SideMenu($value["SUB_MENU"], $cur_menu);
            else:
                $sty = ($cur_menu == $value["URL"])? "class='active'" : "";
                $menustr .= "<li $sty>
                    <a href=\"{$this->_server}{$value["URL"]}\"> $ico
                        <span>{$value["MENU"]}</span>
                    </a>";
            endif;
            
            $menustr .= "</li>"; 
        }
        
        $fixedmenu = ($this->isClient)? $this->_fixed_menu_client : $this->_fixed_menu_admin;
        $menustr .= $fixedmenu . '
                </ul>';
        
        return $menustr;
    }
    
    /**
     * Convert SubMenu Object to Side Menu AdminLTE format
     * 
     * @param mixed $menus Menu Object
     * 
     * @return string Side Menu AdminLTE format string
     */
    private function _submenu2SideMenu($menus, $cur_menu="") {

        $menustr = '
                    <ul class="treeview-menu">';
        foreach ($menus as $key => $value) {
            
            $ico = ($value["MENUICON"] != "")? "<i class=\"{$value["MENUICON"]}\"></i>" : "";
            $subico = (isset($value["SUB_MENU"]) && count($value["SUB_MENU"]))? "<span class=\"pull-right-container\"><i class=\"fa fa-angle-left pull-right\"></i></span>" : "";
            
            if (isset($value["SUB_MENU"]) && count($value["SUB_MENU"])):
                $sty = ($pid == $value["MID"])? "active menu-open" : "";
                $menustr .= "<li class=\"treeview $sty\">
                    <a href=\"{$this->_server}{$value["URL"]}\"> $ico
                        {$value["MENU"]}
                        $subico
                    </a>";
                
                $menustr .= $this->_submenu2SideMenu($value["SUB_MENU"]);
            else:
                $sty = ($cur_menu == $value["URL"])? "class='active'" : "";
                $menustr .= "<li $sty>
                    <a href=\"{$this->_server}{$value["URL"]}\"> $ico
                        {$value["MENU"]}
                    </a>";
            endif;
            
            $menustr .= "</li>"; 
        }
        
        $menustr .= '
                    </ul>
                    ';
        
        return $menustr;
    }


    /**
     * This method fetches and returns the parent menus for the user group(s) the user belongs to
     * 
     * @param string $user_group String of group(s) the user belongs to
     * 
     * @return object An array containing the parent menu(s) the user or user group(s) has access to
     */
    private function _getParentMenus($user_group) {  
        if (is_array($user_group)):
            $params = implode(",", array_fill(0, count($user_group), "?"));
            $types = "ii" . str_repeat("i", count($user_group));
        else:
            $params = "?";
            $types = "iii";
        endif;
        
        $strsql = "SELECT m.MID, m.MENU, m.URL, m.MENUICON, m.MENUIMAGE, m.TOOLTIP, m.PARENTMID, m.MENU_ORDER,m.IS_ACTIVE
            FROM tbl_menus m 
WHERE IS_ACTIVE = 1 AND m.IS_USER_MENU=? 
AND (m.PARENTMID = 0 OR MID IN (SELECT PARENTMID FROM `tbl_usergroup_menus` ugm LEFT JOIN tbl_menus mm ON ugm.MID = mm.MID 
WHERE m.IS_ACTIVE = 1 AND m.IS_USER_MENU=? AND UGID IN ($params)))
ORDER BY m.PARENTMID, m.MENU_ORDER";
        
//        $types = "ii" . str_repeat("i", count($user_group));
//        $args = array_merge( [$strsql, $types, $this->isClient, $this->isClient], $user_group);
        if (is_array($user_group)):
            $args = array_merge( [$strsql, $types, $this->isClient, $this->isClient], $user_group);
        else:
            $args = [$strsql, $types, $this->isClient, $this->isClient, $user_group];
        endif;
        $some_sth = call_user_func_array(array($this->_db, 'select'), $this->ref($args));
        
        //$some_sth = $this->_db->select($strsql, 'iis', $this->isClient, $this->isClient, $user_group);
        $rows = $some_sth->num_rows();
        while ( $row = $some_sth->fetch_object() ) :
            $this->umenus[$row->MENU_ORDER] = ["MID"=>$row->MID, "MENU"=>$row->MENU, "URL"=>$row->URL,
                                                "MENUICON"=>$row->MENUICON, "MENUIMAGE"=>$row->MENUIMAGE, 
                                                "TOOLTIP"=>$row->TOOLTIP, "PARENTMID"=>$row->PARENTMID,
                                                "IS_ACTIVE"=>$row->IS_ACTIVE, "MENU_ORDER"=>$row->MENU_ORDER,
                                                "SUB_MENU" => [] ];
        endwhile;
    }
    
    /**
     * This method fetches and returns the parent menus for Admin Use
     * 
     * @param integer $isActive Whether to select only active menu or all
     * @param integer $isClient Determines if to load Students' menu or that of Admin/Staff. Loads students' menu if set to 1
     * 
     * @return object An array containing the parent menu(s)
     */
    private function _getParentMenusAdmin($isActive=0, $isClient=1) {  
        
        if ($isActive) :
            $strsql = "SELECT m.MID, m.MENU, m.URL, m.MENUICON, m.MENUIMAGE, m.TOOLTIP, m.PARENTMID, m.MENU_ORDER,m.IS_ACTIVE
                FROM tbl_menus m 
    WHERE IS_ACTIVE = ? AND m.IS_USER_MENU=? 
    AND (m.PARENTMID = 0 OR MID IN (SELECT PARENTMID FROM `tbl_usergroup_menus` ugm LEFT JOIN tbl_menus mm ON ugm.MID = mm.MID 
    WHERE m.IS_ACTIVE = ? AND m.IS_USER_MENU=?))
    ORDER BY m.PARENTMID, m.MENU_ORDER";

            $some_sth = $this->_db->select($strsql, 'iiii', $isActive, $isClient, $isActive, $isClient);
        else:
            $strsql = "SELECT m.MID, m.MENU, m.URL, m.MENUICON, m.MENUIMAGE, m.TOOLTIP, m.PARENTMID, m.MENU_ORDER,m.IS_ACTIVE
                FROM tbl_menus m 
    WHERE m.IS_USER_MENU=? 
    AND (m.PARENTMID = 0 OR MID IN (SELECT PARENTMID FROM `tbl_usergroup_menus` ugm LEFT JOIN tbl_menus mm ON ugm.MID = mm.MID 
    WHERE m.IS_USER_MENU=?))
    ORDER BY m.PARENTMID, m.MENU_ORDER";

            $some_sth = $this->_db->select($strsql, 'ii', $isClient, $isClient);
        endif;
        $rows = $some_sth->num_rows();
        while ( $row = $some_sth->fetch_object() ) :
            $this->umenus[$row->MENU_ORDER] = ["MID"=>$row->MID, "MENU"=>$row->MENU, "URL"=>$row->URL,
                                                "MENUICON"=>$row->MENUICON, "MENUIMAGE"=>$row->MENUIMAGE, 
                                                "TOOLTIP"=>$row->TOOLTIP, "PARENTMID"=>$row->PARENTMID,
                                                "IS_ACTIVE"=>$row->IS_ACTIVE, "MENU_ORDER"=>$row->MENU_ORDER,
                                                "SUB_MENU" => [] ];
        endwhile;
    }
    
    public function getParentMenuID($menuurl) {
        $strsql = "SELECT PARENTMID FROM tbl_menus WHERE URL = ?";
        $some_sth = $this->_db->select($strsql, 's', $menuurl);
        if ($some_sth->num_rows()):
            $row = $some_sth->fetch_object();
            return $row->PARENTMID;
        else:
            return 0;
        endif;
    }
    
    /**
     * This method fetches and returns ALL the parent menus for Admin Use
     * 
     * @param integer $isActive Whether to select only active menu or all
     * @param integer $isClient Determines if to load Students' menu or that of Admin/Staff. Loads students' menu if set to 1
     * 
     * @return object An array containing the parent menu(s)
     */
    private function _getAllParentMenusAdmin($isActive=0, $isClient=1) {  
        
        if ($isActive) :
            $strsql = "SELECT DISTINCT m.MID, m.MENU, m.URL, m.MENUICON, m.MENUIMAGE, m.TOOLTIP, m.PARENTMID, m.MENU_ORDER, m.IS_ACTIVE
 FROM tbl_menus m 
    WHERE IS_ACTIVE = ? AND m.IS_USER_MENU=? 
    AND (PARENTMID = 0 OR 
MID IN (SELECT PARENTMID FROM `tbl_usergroup_menus` ugm LEFT JOIN tbl_menus mm ON ugm.MID = mm.MID 
    WHERE IS_ACTIVE = ? AND m.IS_USER_MENU=?))
    ORDER BY m.PARENTMID, m.MENU_ORDER";

            $some_sth = $this->_db->select($strsql, 'iiii', $isActive, $isClient, $isActive, $isClient);
        else:
            $strsql = "SELECT DISTINCT m.MID, m.MENU, m.URL, m.MENUICON, m.MENUIMAGE, m.TOOLTIP, m.PARENTMID, m.MENU_ORDER, m.IS_ACTIVE
 FROM tbl_menus m 
    WHERE m.IS_USER_MENU=? 
    AND (PARENTMID = 0 OR 
MID IN (SELECT PARENTMID FROM `tbl_usergroup_menus` ugm LEFT JOIN tbl_menus mm ON ugm.MID = mm.MID 
    WHERE m.IS_USER_MENU=?))
    ORDER BY m.PARENTMID, m.MENU_ORDER";

            $some_sth = $this->_db->select($strsql, 'ii', $isClient, $isClient);
        endif;
        $rows = $some_sth->num_rows();
        while ( $row = $some_sth->fetch_object() ) :
            $this->umenus[$row->MID] = ["MID"=>$row->MID, "MENU"=>$row->MENU, "URL"=>$row->URL,
                                                "MENUICON"=>$row->MENUICON, "MENUIMAGE"=>$row->MENUIMAGE, 
                                                "TOOLTIP"=>$row->TOOLTIP, "PARENTMID"=>$row->PARENTMID,
                                                "IS_ACTIVE"=>$row->IS_ACTIVE, "MENU_ORDER"=>$row->MENU_ORDER,
                                                "SUB_MENU" => [], "IS_SELECTED" => 0, "PATH" => "top" ];
        //$this->umenus[$row->MENU_ORDER] = 
        endwhile;
    }
    
    /**
     * This method fetches and returns the menus for the user group(s) the user belongs to
     * 
     * @param object $user_group One-Dimensional array of group(s) the user belongs to
     * 
     * @return object An array containing the menu(s) the user or user group(s) has access to
     */
    public function loadMenus($user_group=[]) {
//        $umenus = [];
        $ugrp = implode(", ", $this->getUserGroup($user_group));
        
        //Get Parent Menus
        $this->_getParentMenus($ugrp);
        
        $params = implode(",", array_fill(0, count($user_group), "?"));
        
        foreach ($this->umenus as $key => $menu) {
            $strsql = "SELECT m.MID, m.MENU, m.URL, m.MENUICON, m.MENUIMAGE, m.TOOLTIP, m.PARENTMID, m.MENU_ORDER,m.IS_ACTIVE  
FROM `tbl_usergroup_menus` ugm 
LEFT JOIN tbl_menus m ON ugm.MID = m.MID
LEFT JOIN tbl_menus pm ON m.PARENTMID = pm.MID
WHERE m.PARENTMID=? AND m.IS_ACTIVE = 1 AND m.IS_USER_MENU=? AND UGID IN (?)
ORDER BY m.PARENTMID, m.MENU_ORDER, m.MENU";
            
            $types = "ii" . str_repeat("i", count($user_group));
            $args = array_merge( [$strsql, $types, $menu["MID"], $this->isClient], $user_group);
            $some_sth = call_user_func_array(array($this->_db, 'select'), $this->ref($args));
        
            //$some_sth = $this->_db->select($strsql, 'iis', $menu["MID"], $this->isClient, $user_group);
            $rows = $some_sth->num_rows();
        
            while ( $row = $some_sth->fetch_object() ) :

                $this->umenus[$key]["SUB_MENU"][$row->MENU_ORDER] = ["MID"=>$row->MID, "MENU"=>$row->MENU, "URL"=>$row->URL,
                                                "MENUICON"=>$row->MENUICON, "MENUIMAGE"=>$row->MENUIMAGE, 
                                                "TOOLTIP"=>$row->TOOLTIP, "PARENTMID"=>$row->PARENTMID,
                                                "IS_ACTIVE"=>$row->IS_ACTIVE, "MENU_ORDER"=>$row->MENU_ORDER
                                                 ];
            endwhile;
        }
        
        return $this->umenus;
        
    }
    
    /**
     * This method fetches and returns the menus for Admin Use
     * 
     * @param integer $isActive Whether to select only active menu or all
     * @param integer $isClient Determines if to load Students' menu or that of Admin/Staff. Loads students' menu if set to 1
     * 
     * @return object An array containing the menu(s) for Admin Use
     */
    public function loadMenusAdmin($isActive=0, $isClient=1) {
        
        //Get Parent Menus
        $this->_getParentMenusAdmin($isActive, $isClient);
        
        foreach ($this->umenus as $key => $menu) {
            if ($isActive) :
                $strsql = "SELECT m.MID, m.MENU, m.URL, m.MENUICON, m.MENUIMAGE, m.TOOLTIP, m.PARENTMID, m.MENU_ORDER,m.IS_ACTIVE  
    FROM `tbl_usergroup_menus` ugm 
    LEFT JOIN tbl_menus m ON ugm.MID = m.MID
    LEFT JOIN tbl_menus pm ON m.PARENTMID = pm.MID
    WHERE m.PARENTMID=? AND m.IS_ACTIVE = ? AND m.IS_USER_MENU=?
    ORDER BY m.PARENTMID, m.MENU_ORDER, m.MENU";

                $some_sth = $this->_db->select($strsql, 'iii', $menu["MID"], $isActive, $isClient);
            else:
                $strsql = "SELECT m.MID, m.MENU, m.URL, m.MENUICON, m.MENUIMAGE, m.TOOLTIP, m.PARENTMID, m.MENU_ORDER,m.IS_ACTIVE  
    FROM `tbl_usergroup_menus` ugm 
    LEFT JOIN tbl_menus m ON ugm.MID = m.MID
    LEFT JOIN tbl_menus pm ON m.PARENTMID = pm.MID
    WHERE m.PARENTMID=? AND m.IS_USER_MENU=?
    ORDER BY m.PARENTMID, m.MENU_ORDER, m.MENU";

                $some_sth = $this->_db->select($strsql, 'ii', $menu["MID"], $isClient);
            endif;
            $rows = $some_sth->num_rows();
        
            while ( $row = $some_sth->fetch_object() ) :

                $this->umenus[$key]["SUB_MENU"][$row->MENU_ORDER] = ["MID"=>$row->MID, "MENU"=>$row->MENU, "URL"=>$row->URL,
                                                "MENUICON"=>$row->MENUICON, "MENUIMAGE"=>$row->MENUIMAGE, 
                                                "TOOLTIP"=>$row->TOOLTIP, "PARENTMID"=>$row->PARENTMID,
                                                "IS_ACTIVE"=>$row->IS_ACTIVE, "MENU_ORDER"=>$row->MENU_ORDER
                                                 ];
            endwhile;
        }
        
        return $this->umenus;
        
    }
    
    /**
     * This method fetches and returns All the menus for Admin Use
     * 
     * @param integer $isActive Whether to select only active menu or all
     * @param integer $isClient Determines if to load Students' menu or that of Admin/Staff. Loads students' menu if set to 1
     * @param integer $ugroupid User Group ID
     * 
     * @return object An array containing the menu(s) for Admin Use
     */
    public function loadAllMenusAdmin($isActive=0, $isClient=1, $ugroupid = 0) {
        
        //Get Parent Menus
        $this->_getAllParentMenusAdmin($isActive, $isClient);
        
        //Get Group Menu IDs if $ugroupid > 0
        $grpmid = [];
        if ($ugroupid > 0):
            $grpmid = $this->getGroupMenuIDs($ugroupid);
        endif;
        
        foreach ($this->umenus as $key => $menu) {
            //Mark as selected if MID is in $grpmid
            $this->umenus[$key]["IS_SELECTED"] = (in_array($menu["MID"], $grpmid))? 1 : $this->umenus[$key]["IS_SELECTED"];
            
            if ($isActive) :
                $strsql = "SELECT m.MID, m.MENU, m.URL, m.MENUICON, m.MENUIMAGE, m.TOOLTIP, m.PARENTMID, m.MENU_ORDER,m.IS_ACTIVE  
    FROM tbl_menus m 
    LEFT JOIN tbl_menus pm ON m.PARENTMID = pm.MID
    WHERE m.PARENTMID=? AND m.IS_ACTIVE = ? AND m.IS_USER_MENU=?
    ORDER BY m.PARENTMID, m.MENU_ORDER, m.MENU";

                $some_sth = $this->_db->select($strsql, 'iii', $menu["MID"], $isActive, $isClient);
            else:
                $strsql = "SELECT m.MID, m.MENU, m.URL, m.MENUICON, m.MENUIMAGE, m.TOOLTIP, m.PARENTMID, m.MENU_ORDER,m.IS_ACTIVE  
    FROM tbl_menus m 
    LEFT JOIN tbl_menus pm ON m.PARENTMID = pm.MID
    WHERE m.PARENTMID=? AND m.IS_USER_MENU=?
    ORDER BY m.PARENTMID, m.MENU_ORDER, m.MENU";

                $some_sth = $this->_db->select($strsql, 'ii', $menu["MID"], $isClient);
            endif;
            $rows = $some_sth->num_rows();
        
            while ( $row = $some_sth->fetch_object() ) :

                $this->umenus[$key]["SUB_MENU"][$row->MID] = ["MID"=>$row->MID, "MENU"=>$row->MENU, "URL"=>$row->URL,
                                                "MENUICON"=>$row->MENUICON, "MENUIMAGE"=>$row->MENUIMAGE, 
                                                "TOOLTIP"=>$row->TOOLTIP, "PARENTMID"=>$row->PARENTMID,
                                                "IS_ACTIVE"=>$row->IS_ACTIVE, "MENU_ORDER"=>$row->MENU_ORDER,
                                                "IS_SELECTED" => (in_array($row->MID, $grpmid))? 1 : 0, 
                                                "PATH" => "top/".$menu["MENU"] ];
                //$this->umenus[$key]["SUB_MENU"][$row->MENU_ORDER]
            endwhile;
        }
        
        return $this->umenus;
        
    }
    
    /**
     * Get Group Menu ID
     * @param int $groupid ID of the Group
     * @return Object Array of Group Menu ID
     */
    public function getGroupMenuIDs($groupid) {
        $strsql = "SELECT MID FROM `tbl_usergroup_menus` WHERE UGID=?";
        $some_sth = $this->_db->select($strsql, 'i', $groupid);
        $ret = [];
        while ( $row = $some_sth->fetch_object() ) :
            $ret[] = $row->MID;
        endwhile;
        
        return $ret;
    }
    
    /**
     * Get Group Menu URLs
     * @param int $groupid ID of the Group
     * @return Object Array of Group Menu URLs
     */
    public function getGroupMenuURLs($groupid) {
        $strsql = "SELECT URL FROM `tbl_usergroup_menus` tum
                    LEFT JOIN tbl_menus tm ON tum.MID=tm.MID WHERE UGID=?";
        $some_sth = $this->_db->select($strsql, 'i', $groupid);
        $ret = [];
        while ( $row = $some_sth->fetch_object() ) :
            $ret[] = $row->URL;
        endwhile;
        
        return $ret;
    }
    
    /**
     * Get All Menu as Tree with Group Menu selected
     * 
     * @param integer $isActive Whether to select only active menu or all
     * @param integer $isClient Determines if to load Students' menu or that of Admin/Staff. Loads students' menu if set to 1
     * @param integer $ugroupid User Group ID     * 
     * @param object $ugrp Array of all user groups
     * 
     * @return object An array containing the menu(s) for Admin Use
     */
    public function getAllMenuSelectGrpMenu($isActive=0, $isClient=1, $ugroupid = 0, $ugrp=[]) {
        $allmenu = $this->loadAllMenusAdmin($isActive, $isClient, $ugroupid);
        
        $ret = "";
        if (count($allmenu)):

            foreach ($allmenu as $key => $value) {

                if (count($value["SUB_MENU"])):
                    foreach ($value["SUB_MENU"] as $keyy => $valuey) {
                        $sel = $valuey["IS_SELECTED"]? "selected=\"selected\"" : "";
                        $ret .= "<option value=\"{$valuey["MID"]}\" data-section=\"{$valuey["PATH"]}\" $sel data-index=\"{$valuey["MENU_ORDER"]}\">{$valuey["MENU"]}</option>";
                    }
                else :
                    $sel = $value["IS_SELECTED"]? "selected=\"selected\"" : "";
                    $ret .= "<option value=\"{$value["MID"]}\" data-section=\"{$value["PATH"]}\" $sel data-index=\"{$value["MENU_ORDER"]}\">{$value["MENU"]}</option>";
                endif;
            }
            return json_encode(["status"=>"SUCCESS", "data" => $ret, "ugrp" => $ugrp[$ugroupid]["ADMIN_TYPE"], 
                                "ugrpid" => $ugroupid]);
        else: 
            return json_encode(["status"=>"Error Occured!"]);
        endif;
    }
    
    /**
     * Get All Menu as Tree with Group Menu selected
     * 
     * @param integer $ugroupid User Group ID     * 
     * @param object $ugrpmenu Array of all user groups
     * 
     * @return integer Status 1- Successful; 0-Failed
     */
    public function saveGroupMenu($ugroupid=0, $ugrpmenu=[]) {
        if ($ugroupid == 0 || !count($ugrpmenu)) :
            return 0;
        else:
            $strsql = "SELECT * FROM tbl_usergroup_menus WHERE UGID=?";
            $some_sth = $this->_db->select($strsql, 'i', $ugroupid);
            $rows = $some_sth->num_rows();
            //Delete Existing User Group Menu
            $del = "DELETE FROM tbl_usergroup_menus WHERE UGID=?";
            
            //Create New User Group Menu
            $lop = [];
            foreach ($ugrpmenu as $val) {
                $lop[] = "($ugroupid, $val, {$this->_uid})";
            }
            $lops = join(",", $lop);
            $ins = "INSERT INTO tbl_usergroup_menus (`UGID`, `MID`, `ADDEDBY`) VALUES $lops";
            
            $saved = $this->_db->execute_query($del, "i", $ugroupid);
            if ($saved || $rows == 0):
                $save = $this->_db->execute_query($ins);
                if ($save) {return 1;} else {return 0;}
            else :
                return 0;
            endif;
            
            
        endif;
    }
    
    
    // ----------------------------------------------------------------------------------
    // helper function to turn an array of values into an array of value references
    // necessary because mysqli_stmt::bind_param needs value refereces for no good reason
    function ref($arr) {
        $refs = array();
        foreach ($arr as $key => $val) $refs[$key] = &$arr[$key];
        return $refs;
    }
    
}
